from manim import *

class CNNVisualization(Scene):
    def construct(self):
        cat_image = ImageMobject("cat.png")
        cat_image.scale_to_fit_height(4)
        self.add(cat_image)

        grid_size = 4 / 5
        grid = VGroup(*[Square(side_length=grid_size) for _ in range(25)])
        grid.arrange_in_grid(rows=5, cols=5, buff=0)
        grid.move_to(cat_image)
        self.add(grid)
 
        conv_kernel = VGroup(*[Square(side_length=grid_size) for _ in range(4)])
        conv_kernel.arrange_in_grid(rows=2, cols=2, buff=0)
        conv_kernel.set_fill(BLUE, opacity=0.5).set_stroke(BLUE)
        conv_kernel.align_to(grid[0], direction=LEFT+UP)
        kernel_params = VGroup(*[Tex(r"$\omega_{" + f"{i+1}{j+1}" + r"}$", font_size=24)
                                 for i in range(2) for j in range(2)])
        for param, square in zip(kernel_params, conv_kernel):
            param.move_to(square.get_center())
        conv_kernel.add(kernel_params)
        self.add(conv_kernel)

        feature_map = VGroup(*[Square(side_length=grid_size) for _ in range(16)])
        feature_map.arrange_in_grid(rows=4, cols=4, buff=0)
        feature_map_labels = VGroup(*[Tex(r"$\beta_{" + f"{i+1}{j+1}" + r"}$", font_size=24)
                                      for i in range(4) for j in range(4)])
        feature_map.next_to(cat_image, RIGHT, buff=1)
        self.add(feature_map)

        text = Text("Convoluting", font_size=24)
        text.to_edge(DOWN)
        self.add(text)

        for i in range(4):
            for j in range(4):
                target_position = grid[(i+1)*5 + (j+1)].get_center() - grid_size / 2 * RIGHT - grid_size / 2 * DOWN
                self.play(conv_kernel.animate.move_to(target_position), run_time=0.5)
                feature_map[i*4 + j].set_fill(YELLOW, opacity=0.5)
                feature_map_labels[i*4 + j].move_to(feature_map[i*4 + j].get_center())
                self.add(feature_map_labels[i*4 + j])
                self.wait(0.5)

        self.remove(cat_image, grid, conv_kernel)
        all_elements = VGroup(feature_map, feature_map_labels)
        self.play(all_elements.animate.move_to(ORIGIN))

        pool_kernel = Square(side_length=2 * grid_size, color=RED, fill_opacity=0.5)
        pool_kernel.align_to(feature_map[0], direction=LEFT+UP)
        self.add(pool_kernel)
        self.play(FadeOut(text))
        text = Text("Pooling", font_size=24).to_edge(DOWN)
        self.play(FadeIn(text))


        final_feature_map = VGroup(*[Square(side_length=grid_size) for _ in range(4)])
        final_feature_map.arrange_in_grid(rows=2, cols=2, buff=0)
        final_feature_map_labels = VGroup(*[Tex(r"$\gamma_{" + f"{i+1}{j+1}" + r"}$", font_size=24)
                                            for i in range(2) for j in range(2)])
        final_feature_map.next_to(feature_map, RIGHT, buff=2)
        self.add(final_feature_map)

        for i in range(2):
            for j in range(2):
                target_position = feature_map[i*2*4+j*2+5].get_center() - grid_size / 2 * RIGHT - grid_size / 2 * DOWN
                self.play(pool_kernel.animate.move_to(target_position), run_time=0.5)
                final_feature_map[i*2 + j].set_fill(GREEN, opacity=0.5)
                final_feature_map_labels[i*2 + j].move_to(final_feature_map[i*2 + j].get_center())
                self.add(final_feature_map_labels[i*2 + j])
                self.wait(0.5)

        self.clear()

        original_image = ImageMobject("cat.png")
        original_image.scale_to_fit_height(3)
        original_image.to_edge(LEFT, buff=1)
        self.play(FadeIn(original_image))

        arrow = Arrow(original_image.get_right(), original_image.get_right() + RIGHT * 1.5, buff=0.2, stroke_width=3).set_color(WHITE)
        arrow.next_to(original_image, RIGHT, buff=0.1)
        self.play(FadeIn(arrow))

        cat_cnn1 = ImageMobject("cat_cnn1.png")
        cat_cnn2 = ImageMobject("cat_cnn2.png")

        cat_cnn1.scale_to_fit_height(3)
        cat_cnn2.scale_to_fit_height(3)

        cat_cnn1.next_to(arrow, RIGHT, buff=0.5)
        cat_cnn2.next_to(cat_cnn1, RIGHT, buff=1)

        self.play(FadeIn(cat_cnn1), FadeIn(cat_cnn2))

        text = Text("Images generated by convolving\nmultiple layers from the original image", font_size=24, line_spacing=1.5)
        text.to_edge(DOWN, buff=0.5)
        self.play(FadeIn(text))

        self.wait(4)

        self.play(FadeOut(original_image), FadeOut(arrow), FadeOut(cat_cnn1), FadeOut(cat_cnn2), FadeOut(text))

if __name__ == "__main__":
    scene = CNNVisualization()
    scene.render()
